<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Marketplace
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */
namespace Webkul\Marketplace\Model\Notification;

use Magento\Customer\Model\Session as CustomerSession;
use Magento\Framework\Data\Form\FormKey;
use Magento\Framework\UrlInterface;
use Magento\Framework\App\Config\ScopeConfigInterface;

class MarketplaceConfigProvider
{

    /**
     * @var CustomerSession
     */
    private $authSession;

    /**
     * @var FormKey
     */
    protected $formKey;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Store\Model\StoreManagerInterface
     */
    protected $storeManager;

    /**
     * @var UrlInterface
     */
    protected $urlBuilder;
    /**
     * @var UrlInterface
     */
    protected $helper;

    /**
     * @var \Magento\Framework\View\Asset\Repository
     */
    protected $viewFileSystem;

     /**
      * @var \Magento\Framework\Stdlib\DateTime\DateTime
      */
    protected $date;

     /**
      * @var \Webkul\Marketplace\Model\ProductFactory
      */
    protected $productModel;

     /**
      * @var \Webkul\Marketplace\Model\SellerFactory
      */
    protected $sellerModel;

     /**
      * @var \Magento\Customer\Model\CustomerFactory
      */
    protected $customerModel;

     /**
      * @var \Magento\Catalog\Model\ProductFactory
      */
    protected $productFactory;

     /**
      * @var \Webkul\Marketplace\Model\FeedbackFactory
      */
    protected $feedbackFactory;
    /**
     * @var \Magento\Backend\Helper\Data
     */
    protected $adminHelper;
    /**
     * @var \Webkul\Marketplace\Helper\Orders
     */
    protected $orderHelper;

    /**
     * Construct
     *
     * @param \Magento\Backend\Model\Auth\Session $authSession
     * @param FormKey $formKey
     * @param ScopeConfigInterface $scopeConfig
     * @param \Magento\Store\Model\StoreManagerInterface $storeManager
     * @param UrlInterface $urlBuilder
     * @param \Webkul\Marketplace\Helper\Data $helper
     * @param \Webkul\Marketplace\Helper\Orders $orderHelper
     * @param \Magento\Framework\Stdlib\DateTime\DateTime $date
     * @param \Magento\Framework\View\Asset\Repository $viewFileSystem
     * @param \Magento\Backend\Helper\Data $adminHelper
     * @param \Webkul\Marketplace\Model\ProductFactory $productModel
     * @param \Webkul\Marketplace\Model\SellerFactory $sellerModel
     * @param \Magento\Customer\Model\CustomerFactory $customerModel
     * @param \Magento\Catalog\Model\ProductFactory $productFactory
     * @param \Webkul\Marketplace\Model\FeedbackFactory $feedbackFactory
     */
    public function __construct(
        \Magento\Backend\Model\Auth\Session $authSession,
        FormKey $formKey,
        ScopeConfigInterface $scopeConfig,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        UrlInterface $urlBuilder,
        \Webkul\Marketplace\Helper\Data $helper,
        \Webkul\Marketplace\Helper\Orders $orderHelper,
        \Magento\Framework\Stdlib\DateTime\DateTime $date,
        \Magento\Framework\View\Asset\Repository $viewFileSystem,
        \Magento\Backend\Helper\Data $adminHelper,
        \Webkul\Marketplace\Model\ProductFactory $productModel,
        \Webkul\Marketplace\Model\SellerFactory $sellerModel,
        \Magento\Customer\Model\CustomerFactory $customerModel,
        \Magento\Catalog\Model\ProductFactory $productFactory,
        \Webkul\Marketplace\Model\FeedbackFactory $feedbackFactory
    ) {
        $this->authSession = $authSession;
        $this->formKey = $formKey;
        $this->scopeConfig = $scopeConfig;
        $this->storeManager = $storeManager;
        $this->urlBuilder = $urlBuilder;
        $this->helper = $helper;
        $this->adminHelper = $adminHelper;
        $this->orderHelper = $orderHelper;
        $this->viewFileSystem = $viewFileSystem;
        $this->date = $date;
        $this->productModel = $productModel;
        $this->sellerModel = $sellerModel;
        $this->customerModel = $customerModel;
        $this->productFactory = $productFactory;
        $this->feedbackFactory = $feedbackFactory;
    }
    /**
     * Get config data
     *
     * @return array
     */
    public function getConfig()
    {
        $output = [];
        if ($this->isAdminLoggedIn()) {
            $defaultImageUrl = $this->viewFileSystem->getUrlWithParams(
                'Webkul_Marketplace::images/icons_notifications.png',
                []
            );
            $output['formKey'] = $this->formKey->getFormKey();
            $output['image'] = $defaultImageUrl;
            $output['productNotification'] = $this->getProductNotificationData();
            $output['sellerNotification'] = $this->getSellerNotificationData();
            $output['feedbackNotification'] = $this->getFeedbackNotificationData();
        }
        return $output;
    }

    /**
     * Return seller product data for notification.
     *
     * @return array
     */
    protected function getProductNotificationData()
    {
        $productData = [];
        $markeplaceProduct = $this->productModel->create()->getCollection()
        ->addFieldToFilter('admin_pending_notification', ['neq' => 0]);

        if ($markeplaceProduct->getSize()) {
            foreach ($markeplaceProduct as $value) {
                $title = '';
                $desc = '';
                if ($value->getAdminPendingNotification() == 1) {
                    $title = __('New product added by the seller');
                    $desc = __(
                        sprintf(
                            'Product "%s" added by the seller "%s", please check Marketplace product list.',
                            '<span class="wk-focus">'.$this->getProductName($value->getMageproductId()).'</span>',
                            '<span class="wk-focus">'.$this->getSellerName($value->getSellerId())->getName().'</span>'
                        )
                    );
                }
                if ($value->getAdminPendingNotification() == 2) {
                    $title = __('Product details updated by seller');
                    $desc = __(
                        sprintf(
                            'Product "%s" updated by the seller "%s", please check Marketplace product list.',
                            '<span class="wk-focus">'.$this->getProductName($value->getMageproductId()).'</span>',
                            '<span class="wk-focus">'.$this->getSellerName($value->getSellerId())->getName().'</span>'
                        )
                    );
                }
                $productData[] = [
                    'product_id' => $value->getMageproductId(),
                    'title' => $title,
                    'desc'  => $desc,
                    'seller_id' => $value->getSellerId(),
                    'seller_name' => $this->getSellerName($value->getSellerId())->getName(),
                    'updated_time'  => $this->date->gmtDate(
                        'l jS \of F Y h:i:s A',
                        strtotime($value->getUpdatedAt())
                    ),
                    'url' => $this->adminHelper->getUrl('marketplace/product')
                ];
            }
        }

        return $productData;
    }

    /**
     * Create newly created seller notification data.
     *
     * @return array
     */
    protected function getSellerNotificationData()
    {
        $sellerData = [];
        $sellerCollection = $this->sellerModel->create()->getCollection()
        ->addFieldToFilter('admin_notification', ['neq' => 0]);
        if ($sellerCollection->getSize()) {
            foreach ($sellerCollection as $seller) {
                $title = '';
                $desc = '';
                if ($seller->getAdminNotification() == 1) {
                    $title = __('New Customer registered as seller');
                    $desc = __(
                        sprintf(
                            'Customer "%s" requested to become seller, click here to see Marketplce Seller list.',
                            '<span class="wk-focus">'.$this->getSellerName($seller->getSellerId())->getName().'</span>'
                        )
                    );
                }
                $sellerData[] = [
                    'seller_id' => $seller->getSellerId(),
                    'title' => $title,
                    'desc'  => $desc,
                    'seller_name' => $this->getSellerName($seller->getSellerId())->getName(),
                    'updated_time'  => $this->date->gmtDate(
                        'l jS \of F Y h:i:s A',
                        strtotime($seller->getUpdatedAt())
                    ),
                    'url' => $this->adminHelper->getUrl('marketplace/seller')
                ];
            }
        }
        return $sellerData;
    }

    /**
     * Create newly created seller notification data.
     *
     * @return array
     */
    protected function getFeedbackNotificationData()
    {
        $feedBackData = [];
        $feedBackCollection = $this->feedbackFactory->create()
        ->getCollection()
        ->addFieldToFilter('admin_notification', ['neq' => 0]);
        if ($feedBackCollection->getSize()) {
            foreach ($feedBackCollection as $feedback) {
                $title = '';
                $desc = '';
                if ($feedback->getAdminNotification() == 1) {
                    $title = __('New review submitted by buyer');
                    $desc = __(
                        sprintf(
                            'Buyer submitted review for seller "%s", click here to see the feedback list',
                            '<span class="wk-focus">'.
                            $this->getSellerName($feedback->getSellerId())->getName().'</span>'
                        )
                    );
                }
                $feedBackData[] = [
                    'seller_id' => $feedback->getSellerId(),
                    'title' => $title,
                    'desc'  => $desc,
                    'seller_name' => $this->getSellerName($feedback->getSellerId())->getName(),
                    'updated_time'  => $this->date->gmtDate(
                        'l jS \of F Y h:i:s A',
                        strtotime($feedback->getCreatedAt())
                    ),
                    'url' => $this->adminHelper->getUrl('marketplace/feedback')
                ];
            }
        }
        return $feedBackData;
    }

    /**
     * Product name
     *
     * @param  int $productId
     * @return string
     */
    protected function getProductName($productId)
    {
        $product = $this->productFactory->create()->load($productId);

        return $product->getName();
    }

    /**
     * Load customer by id
     *
     * @param  int $id
     * @return Magento\Customer\Model\Customer
     */
    protected function getSellerName($id)
    {
        $customer = $this->customerModel->create()->load($id);
        return $customer;
    }

    /**
     * Check if customer is logged in
     *
     * @return bool
     */
    private function isAdminLoggedIn()
    {
        return (bool)$this->authSession->isLoggedIn();
    }
}
