<?php
/**
 * Webkul Software.
 *
 * @category  Webkul
 * @package   Webkul_Marketplace
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\Marketplace\Block\Product\Helper\Form\Gallery;

use Magento\Catalog\Model\Product;

class Content extends \Magento\Framework\View\Element\Template
{
    /**
     * @var \Magento\Catalog\Model\Product
     */
    protected $_product;

    /**
     * @var \Magento\Catalog\Model\Product\Media\Config
     */
    protected $_mediaConfig;

    /**
     * @var \Magento\Framework\File\Size
     */
    protected $_fileSizeService;

    /**
     * @var \Magento\Framework\Json\EncoderInterface
     */
    protected $_jsonEncoderInterface;

    /**
     * @var Registry
     */
    protected $_coreRegistry = null;
    /**
     * @var \Webkul\Marketplace\Helper\Data
     */
    protected $_marketplaceHelper;

    /**
     * @param \Magento\Backend\Block\Template\Context     $context
     * @param \Magento\Catalog\Model\Product\Media\Config $mediaConfig
     * @param \Magento\Framework\File\Size                $fileSize
     * @param \Magento\Framework\Json\EncoderInterface    $jsonEncoderInterface
     * @param \Magento\Framework\Registry                 $coreRegistry
     * @param Product                                     $product
     * @param \Webkul\Marketplace\Helper\Data             $marketplaceHelper
     * @param array                                       $data
     */
    public function __construct(
        \Magento\Framework\View\Element\Template\Context $context,
        \Magento\Catalog\Model\Product\Media\Config $mediaConfig,
        \Magento\Framework\File\Size $fileSize,
        \Magento\Framework\Json\EncoderInterface $jsonEncoderInterface,
        \Magento\Framework\Registry $coreRegistry,
        Product $product,
        \Webkul\Marketplace\Helper\Data $marketplaceHelper,
        array $data = [],
    ) {
        $this->_product = $product;
        $this->_mediaConfig = $mediaConfig;
        $this->_fileSizeService = $fileSize;
        $this->_jsonEncoderInterface = $jsonEncoderInterface;
        $this->_coreRegistry = $coreRegistry;
        $this->_marketplaceHelper = $marketplaceHelper;
        parent::__construct($context, $data);
    }

    /**
     * Get file size service
     *
     * @return \Magento\Framework\File\Size
     */
    public function getFileSizeService()
    {
        return $this->_fileSizeService;
    }

    /**
     * Retrieve product.
     *
     * @return \Magento\Catalog\Model\Product
     */
    public function getProduct()
    {
        return $this->_coreRegistry->registry('current_product');
    }

    /**
     * Get product image data.
     *
     * @return array
     */
    public function getProductImagesJson()
    {
        $productColl = $this->getProduct();
        if ($productColl) {
            $mediaGalleryImages = $productColl->getData("media_gallery");
            $productImages = [];
            if (isset($mediaGalleryImages["images"]) && count($mediaGalleryImages["images"]) > 0) {
                $mediaGalleryImages = $mediaGalleryImages["images"];
                foreach ($mediaGalleryImages as $key => &$mediaGalleryImage) {
                    $mediaGalleryImage['url'] = $this->_mediaConfig->getMediaUrl(
                        $mediaGalleryImage["file"]
                    );
                    array_push($productImages, $mediaGalleryImage);
                }
                return $this->_jsonEncoderInterface->encode($productImages);
            }
        }
        return '[]';
    }

    /**
     * Get product image types
     *
     * @return array
     */
    public function getProductImageTypes()
    {
        $productImageTypes = [];
        $productColl = $this->getProduct();
        foreach ($this->getProductMediaAttributes() as $attribute) {
            $productImageTypes[$attribute->getAttributeCode()] = [
              'code' => $attribute->getAttributeCode(),
              'value' => $productColl[$attribute->getAttributeCode()],
              'label' => $attribute->getFrontend()->getLabel(),
              'name' => 'product['.$attribute->getAttributeCode().']',
            ];
        }

        return $productImageTypes;
    }

    /**
     * Get media attribute
     *
     * @return array
     */
    public function getProductMediaAttributes()
    {
        $mediaAttributes = [];
        $allowedMediaAttributes = $this->getAllowedMediaAttributes();
        $productMediaAttributes = $this->_product->getMediaAttributes();
        foreach ($productMediaAttributes as $attribute) {
            if (in_array($attribute->getAttributeCode(), $allowedMediaAttributes)) {
                $mediaAttributes[$attribute->getAttributeCode()] = $attribute;
            }
        }
        return $mediaAttributes;
    }

    /**
     * Get Allowed Media Attributes returns the allowed media attributes]
     *
     * @return array
     */
    public function getAllowedMediaAttributes()
    {
        return ['image', 'small_image', 'thumbnail', 'swatch_image'];
    }
    /**
     * Get Draft product data
     *
     * @param int $draftId
     * @return array
     */
    public function getDraftProduct($draftId = 0)
    {
        $draftId = (!$draftId)?$this->getRequest()->getParam("draft_id"):0;
        if (!$draftId) {
            return [];
        }
        $draftProd = $this->_marketplaceHelper->getDraftProductCollection($draftId);
        $draftProdData = $this->_marketplaceHelper->jsonToArray($draftProd->getContent());
        return $draftProdData;
    }
}
