<?php

namespace Webkul\GadzmoneyWallet\Rewrite\Plugin;

use Magento\Sales\Api\Data\OrderInterface;
use Magento\Sales\Api\OrderManagementInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCryptoTransactions;

/**
 * Class OrderManagement plugin to change information
 */
class OrderManagement extends \Webkul\BitcoinCryptoPayment\Plugin\OrderManagement
{
    /**
     * @var \Webkul\GadzmoneyWallet\Helper\Data
     */
    protected $helper;

    /**
     * Constructor
     *
     * @param OrderRepositoryInterface $orderRepository
     * @param \Magento\Quote\Model\QuoteRepository $quoteRepository
     * @param BitcoinCryptoTransactions $cryptoTranactions
     * @param \Webkul\BitcoinCryptoPayment\Helper\Email $emailHelper
     * @param \Magento\Framework\App\ResourceConnection $resourceconnection
     * @param \Webkul\GadzmoneyWallet\Helper\Data $helper
     */
    public function __construct(
        OrderRepositoryInterface $orderRepository,
        \Magento\Quote\Model\QuoteRepository $quoteRepository,
        BitcoinCryptoTransactions $cryptoTranactions,
        \Webkul\BitcoinCryptoPayment\Helper\Email $emailHelper,
        \Magento\Framework\App\ResourceConnection $resourceconnection,
        \Webkul\GadzmoneyWallet\Helper\Data $helper
    ) {
        $this->helper = $helper;
        parent::__construct(
            $orderRepository,
            $quoteRepository,
            $cryptoTranactions,
            $emailHelper,
            $resourceconnection
        );
    }

    /**
     * After Place
     *
     * @param OrderManagementInterface $subject
     * @param OrderInterface $result
     *
     * @return OrderInterface
     */
    public function afterPlace(
        OrderManagementInterface $subject,
        OrderInterface $result
    ) {
        $orderId = $result->getId();
        if ($orderId) {
            $payment = $result->getPayment();
            $method = $payment->getMethodInstance()->getCode();
            if ($method == BitcoinCrypto::PAYMENT_METHOD_CRYPTOBITCOIN_CODE) {
                $quoteId = $result->getQuoteId();
                $quote = $this->quoteRepository->get($quoteId);
                $order = $this->_orderRepository->get($orderId);
                $order->setState('awaiting_for_crypto_payment')
                    ->setStatus('awaiting_for_crypto_payment');
                $crypto_currency = $quote->getData('crypto_currency') ?? '';
                $conversion_rate = $quote->getData('conversion_rate') ?? '';
                $crypto_id = $quote->getData('crypto_payment_method_id') ?? '';
                $crypto_amount = $quote->getData('crypto_amount') ?? '';
                $order->setData('crypto_amount', $crypto_amount);
                $order->setCryptoCurrency($crypto_currency);
                $order->setConversionRate($conversion_rate);
                $order->setData('crypto_payment_method_id', $crypto_id);
                $this->_orderRepository->save($order);

                $trans = $this->cryptoTranactions->setSalesOrderId($orderId);
                $this->cryptoTranactions->setCryptoAmount($crypto_amount);
                $this->cryptoTranactions->setReminderCount(0);
                $this->cryptoTranactions->setCryptoPaymentMethodId($crypto_id);
                $this->cryptoTranactions->setCreatedAt(date("Y-m-d h:i:sa"));
                $trans->save();

                $crypto_payments = $this->resourceconnection
                    ->getTableName('webkul_bitcoin_crypto_payments');
                $transactionDetails = $this->cryptoTranactions->getCollection();
                $orderFirstItem = $order->getItems()[0];
                $orderedFirstProductId = $orderFirstItem->getProductId();
                $transactionDetails->addFieldToFilter('sales_order_id', ['in' => $order->getId()]);
                $cryptoPaymentId = $transactionDetails->getFirstItem()->getCryptoPaymentMethodId();
                if($cryptoPaymentId == 0) {
                    $transactioData = $transactionDetails->getData();
                    $transactioData[0]["wallet_name"] = "Gadzmoney";
                    $transactioData[0]["wallet_address"] = $this->helper->getAlphWalletAddress($orderedFirstProductId);
                    $transactioData[0]["is_tag_required"] = "0";
                    $transactioData[0]["destination_tag_memo"] = null;
                } else {
                    $transactionDetails->getSelect()->join(
                        ['crypto_payments' => $crypto_payments],
                        'main_table.crypto_payment_method_id = crypto_payments.entity_id',
                        ['wallet_name' => 'crypto_payments.name',
                            'wallet_address' => 'crypto_payments.address',
                            'is_tag_required' => 'crypto_payments.is_tag_required',
                            'destination_tag_memo' => 'crypto_payments.destination_tag_memo']
                    );
                    $transactioData = $transactionDetails->getData();
                }

                $emailTemplateVariables = [
                    'customer_name' => $order->getCustomerName() ?? 'Guest',
                    'order_incrementid' => $order->getIncrementId(),
                    'wallet_name' => $transactioData[0]['wallet_name'] ?? '',
                    'wallet_address' => $transactioData[0]['wallet_address'] ?? '',
                    'crypto_amount' => $transactioData[0]['crypto_amount'] ?? '',
                    'order_amount' => $this->emailHelper->currencyFormat(
                        $order->getGrandTotal(),
                        $order->getOrderCurrencyCode()
                    ),
                    'is_tag_required' => $transactioData[0]['is_tag_required'] ?? "NA",
                    'destination_tag_memo' => $transactioData[0]['destination_tag_memo'] ?? "",
                ];
                $template = 'webkul/custom/email_template_payment';
                $receiverInfo = [$order->getCustomerEmail()];
                $this->emailHelper->sendEmail($emailTemplateVariables, $receiverInfo, $template);
            }
        }
        return $result;
    }
}
