<?php

namespace Webkul\GadzmoneyWallet\Controller\Crypto;

use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\Controller\ResultFactory;
use Magento\Sales\Api\OrderRepositoryInterface;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCryptoTransactions;
use Magento\Customer\Model\CustomerFactory;


/**
 * Class SaveTransaction
 * Generate request data to create order and proceed payment
 */
class SaveTransaction extends \Magento\Framework\App\Action\Action
{
    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    protected $resourceconnection;

    /**
     * @var BitcoinCryptoTransactions
     */
    protected $cryptoTranactions;

    /**
     * @var OrderRepositoryInterface
     */
    protected $_orderRepository;

    /**
     * @var \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto
     */
    protected $configsection;

    /**
     * @var ScopeConfigInterface
     */
    protected $scopeConfigInterface;

    /**
     * @var \Magento\Sales\Model\Service\InvoiceService
     */
    protected $invoiceService;

    /**
     * @var \Magento\Sales\Model\Order\Email\Sender\InvoiceSender
     */
    protected $invoiceSender;

    /**
     * @var \Magento\Framework\DB\Transaction
     */
    protected $transaction;

    /**
     * @var \Magento\Sales\Model\Order\Email\Sender\OrderSender
     */
    protected $orderSender;

    /**
     * @var \Webkul\BitcoinCryptoPayment\Logger\Logger
     */
    protected $logger;

    /**
     * @var \Webkul\BitcoinCryptoPayment\Helper\Email
     */
    protected $emailhelper;

    /**
     * @var \Webkul\Marketplace\Model\OrdersFactory
     */
    protected $mpOrdersFactory;

    /**
     * @var \Magento\Customer\Model\CustomerFactory
     */
    protected $customerModel;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Action\Context $context
     * @param \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory
     * @param \Magento\Framework\App\ResourceConnection $resourceconnection
     * @param BitcoinCryptoTransactions $cryptoTranactions
     * @param OrderRepositoryInterface $_orderRepository
     * @param \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto $configsection
     * @param ScopeConfigInterface $scopeConfigInterface
     * @param \Magento\Sales\Model\Service\InvoiceService $invoiceService
     * @param \Magento\Sales\Model\Order\Email\Sender\InvoiceSender $invoiceSender
     * @param \Magento\Framework\DB\Transaction $transaction
     * @param \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender
     * @param \Webkul\BitcoinCryptoPayment\Logger\Logger $logger
     * @param \Webkul\BitcoinCryptoPayment\Helper\Email $emailhelper
     */
    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory,
        \Magento\Framework\App\ResourceConnection $resourceconnection,
        BitcoinCryptoTransactions $cryptoTranactions,
        OrderRepositoryInterface $_orderRepository,
        \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto $configsection,
        ScopeConfigInterface $scopeConfigInterface,
        \Magento\Sales\Model\Service\InvoiceService $invoiceService,
        \Magento\Sales\Model\Order\Email\Sender\InvoiceSender $invoiceSender,
        \Magento\Framework\DB\Transaction $transaction,
        \Magento\Sales\Model\Order\Email\Sender\OrderSender $orderSender,
        \Webkul\BitcoinCryptoPayment\Logger\Logger $logger,
        \Webkul\BitcoinCryptoPayment\Helper\Email $emailhelper,
        \Webkul\Marketplace\Model\OrdersFactory $mpOrdersFactory,
        CustomerFactory $customerModel
    ) {
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->resourceconnection = $resourceconnection;
        $this->cryptoTranactions = $cryptoTranactions;
        $this->_orderRepository = $_orderRepository;
        $this->configsection = $configsection;
        $this->scopeConfigInterface = $scopeConfigInterface;
        $this->invoiceService = $invoiceService;
        $this->invoiceSender = $invoiceSender;
        $this->transaction = $transaction;
        $this->orderSender = $orderSender;
        $this->logger = $logger;
        $this->emailhelper = $emailhelper;
        $this->mpOrdersFactory = $mpOrdersFactory;
        $this->customerModel = $customerModel;
        parent::__construct($context);
    }

    /**
     * Execute function
     *
     * @return array
     */
    public function execute()
    {
        $idCryptoOrder = $this->getRequest()->getParam('idCryptoOrder');
        $idCryptoTxn = $this->getRequest()->getParam('idCryptoTxn');

        $responseContent = [
            'success' => false,
            'message' => __('Order Id or Transaction Id not found.'),
        ];

        if ($idCryptoOrder && $idCryptoTxn) {
            if (!$idCryptoOrder) {
                $responseContent = [
                    'success' => false,
                    'message' => __('Invalid order ID'),
                ];
                $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
                $response->setData($responseContent);
                return $response;
            }
            if (strlen(trim($idCryptoTxn)) == 0) {
                $responseContent = [
                    'success' => false,
                    'message' => __('Transaction ID field is required.'),
                ];
                $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
                $response->setData($responseContent);
                return $response;
            }
            // elseif (strlen(trim($idCryptoTxn)) != 66) {
            //     $responseContent = [
            //         'success' => false,
            //         'message' => __('Transaction ID must be 66 characters long.'),
            //     ];
            //     $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
            //     $response->setData($responseContent);
            //     return $response;
            // }
             elseif (!$this->isGenericName($idCryptoTxn)) {
                $responseContent = [
                    'success' => false,
                    'message' => __('Invalid Transaction ID.'),
                ];
                $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
                $response->setData($responseContent);
                return $response;
            }
            $objCryptOrder = $this->getOrderDetailsForCryro((int) $idCryptoOrder);
            if (!$objCryptOrder->getEntityId()) {

                $responseContent = [
                    'success' => false,
                    'message' => __('Crypto order ID does not exists.'),
                ];
                $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
                $response->setData($responseContent);
                return $response;
            }

            $crTrans = $this->cryptoTranactions->load($idCryptoTxn, 'transaction_number');
            if ($crTrans->getId()) {
                $responseContent = [
                    'success' => false,
                    'message' => __('Transaction Id already exists. Please use other transaction Id.'),
                ];
                $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
                $response->setData($responseContent);
                return $response;
            }

            if ($objCryptOrder->getStatus() == 'awaiting_for_crypto_payment') {
                $idCryptoTxn = trim($idCryptoTxn);
                $trans = $this->cryptoTranactions->load($objCryptOrder->getEntityId(), 'sales_order_id');
                $trans->setTransactionNumber($idCryptoTxn);
                $trans->setUpdatedAt(date("Y-m-d h:i:sa"));

                if ($trans->save()) {

                    // Update order status and send mail to customer
                    $order = $this->_orderRepository->get($objCryptOrder->getEntityId());
                    $order->setState('verifying_crypto_payment')->setStatus('verifying_crypto_payment');
                    $this->_orderRepository->save($order);

                    // Update transaction in order
                    $this->processPayment($idCryptoTxn, $order);

                    // Send mail to seller
                    $sellerOrder = $this->mpOrdersFactory->create()->getCollection()->addFieldToFilter('order_id', $order->getId())->getFirstItem();
                    if ($sellerOrder->getId()) {
                        $this->sendMailToSeller($order, $sellerOrder->getSellerId());
                    } else {
                        // Send mail to admin
                        $adminEmail = $this->scopeConfigInterface->getValue(
                            'trans_email/ident_support/email',
                            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
                        );
    
                        if ($this->configsection->isMailToAdminOnConfirmation() && $adminEmail) {
                            $this->sendMailToAdmin($order);
                        }
                    }


                    $responseContent = [
                        'success' => true,
                        'message' => __('Transaction ID successfully sent to admin for verification!'),
                    ];
                }
            }
        }

        $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        $response->setData($responseContent);
        return $response;
    }

    /**
     * Is Generic
     *
     * @param string $name
     */
    public function isGenericName($name)
    {
        return empty($name) || preg_match('/^[^<>={}]*$/u', $name);
    }

    /**
     * Get OrderDetails
     *
     * @param int $orderId
     * @return \Magento\Sales\Model\ResourceModel\Order\Collection
     */
    public function getOrderDetailsForCryro($orderId)
    {
        $crypto_payments_transactions = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments_transactions');
        $orders = $this->orderCollectionFactory->create();
        $orders->addAttributeToSelect('entity_id');
        $orders->addAttributeToSelect('grand_total');
        $orders->addAttributeToSelect('order_currency_code');
        $orders->addAttributeToSelect('crypto_payment_method_id');
        $orders->addAttributeToSelect('status');
        $orders->addFieldToFilter('entity_id', ['in' => $orderId]);
        $orders->getSelect()->join(
            ['crypto_transaction' => $crypto_payments_transactions],
            'main_table.entity_id = crypto_transaction.sales_order_id',
            ['order_id' => 'main_table.entity_id',
                'reminder_count' => 'crypto_transaction.reminder_count',
                'crypto_amount' => 'crypto_transaction.crypto_amount',
                'transaction_number' => 'crypto_transaction.transaction_number']
        );
        return $orders->getFirstItem();
    }

    /**
     * Send payment confirmation to admin
     *
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     *
     * @return void
     */
    public function sendMailToAdmin($order)
    {
        $crypto_payments_transactions = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments_transactions');
        $crypto_payments = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments');
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentTitle = $method->getTitle();
        $orders = $this->orderCollectionFactory->create();
        $orders->addFieldToFilter('entity_id', ['in' => $order->getId()]);
        $orders->getSelect()->joinLeft(
            ['crypto_transaction' => $crypto_payments_transactions],
            'main_table.entity_id = crypto_transaction.sales_order_id',
            ['order_id' => 'main_table.entity_id',
                'reminder_count' => 'crypto_transaction.reminder_count',
                'crypto_amount' => 'crypto_transaction.crypto_amount',
                'transaction_number' => 'crypto_transaction.transaction_number']
        );

        $orders->getSelect()->joinLeft(
            ['crypto_payment' => $crypto_payments],
            'crypto_payment.entity_id = crypto_transaction.crypto_payment_method_id',
            ['wallet_address' => 'crypto_payment.address',
                'wallet_name' => 'crypto_payment.name',
                'wallet_unit' => 'crypto_payment.unit']
        );
        $orderDetails = $orders->getFirstItem();

        // Crypto payment confirmed by customer
        $emailTemplateVariables = [
            'customer_name' => $order->getCustomerName() ?? 'Guest',
            'order_incrementid' => $order->getIncrementId(),
            'wallet_name' => $paymentTitle . '(' . $orderDetails->getWalletName() . ')',
            'wallet_address' => $orderDetails->getWalletAddress(),
            'crypto_amount' => $orderDetails->getCryptoAmount() . ' ' . $orderDetails->getWalletUnit(),
            'order_amount' => $this->emailhelper->currencyFormat(
                $order->getGrandTotal(),
                $order->getOrderCurrencyCode()
            ),
            'transaction_number' => $orderDetails->getTransactionNumber(),
        ];

        $adminEmail = $this->scopeConfigInterface->getValue(
            'trans_email/ident_support/email',
            \Magento\Store\Model\ScopeInterface::SCOPE_STORE
        );

        $receiverInfo = [$adminEmail];
        $template = 'webkul/custom/email_template_transaction';
        $this->emailhelper->sendEmail($emailTemplateVariables, $receiverInfo, $template);
    }

    /**
     * Send payment confirmation to seller
     *
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     *
     * @return void
     */
    public function sendMailToSeller($order, $sellerId)
    {
        $crypto_payments_transactions = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments_transactions');
        $crypto_payments = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments');
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentTitle = $method->getTitle();
        $orders = $this->orderCollectionFactory->create();
        $orders->addFieldToFilter('entity_id', ['in' => $order->getId()]);
        $orders->getSelect()->joinLeft(
            ['crypto_transaction' => $crypto_payments_transactions],
            'main_table.entity_id = crypto_transaction.sales_order_id',
            ['order_id' => 'main_table.entity_id',
                'reminder_count' => 'crypto_transaction.reminder_count',
                'crypto_amount' => 'crypto_transaction.crypto_amount',
                'transaction_number' => 'crypto_transaction.transaction_number']
        );

        $orders->getSelect()->joinLeft(
            ['crypto_payment' => $crypto_payments],
            'crypto_payment.entity_id = crypto_transaction.crypto_payment_method_id',
            ['wallet_address' => 'crypto_payment.address',
                'wallet_name' => 'crypto_payment.name',
                'wallet_unit' => 'crypto_payment.unit']
        );
        $orderDetails = $orders->getFirstItem();

        // Crypto payment confirmed by customer
        $emailTemplateVariables = [
            'customer_name' => $order->getCustomerName() ?? 'Guest',
            'order_incrementid' => $order->getIncrementId(),
            'wallet_name' => $paymentTitle . '(' . $orderDetails->getWalletName() . ')',
            'wallet_address' => $orderDetails->getWalletAddress(),
            'crypto_amount' => $orderDetails->getCryptoAmount() . ' ' . $orderDetails->getWalletUnit(),
            'order_amount' => $this->emailhelper->currencyFormat(
                $order->getGrandTotal(),
                $order->getOrderCurrencyCode()
            ),
            'transaction_number' => $orderDetails->getTransactionNumber(),
        ];

        $customer = $this->customerModel->create()->load($sellerId);

        $sellerEmail = $customer->getEmail();

        $receiverInfo = [$sellerEmail];
        $template = 'webkul/custom/seller_email_template_transaction';
        $this->emailhelper->sendEmail($emailTemplateVariables, $receiverInfo, $template);
    }

    /**
     * Process Payment
     *
     * @param string $transactionId
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @return void
     */
    protected function processPayment($transactionId, $order)
    {
        $orderAmount = round($order->getGrandTotal(), 2);
        $order->setState('verifying_crypto_payment')->setStatus('verifying_crypto_payment');
        $payment = $order->getPayment();

        $payment->setAmountPaid($orderAmount)
            ->setLastTransId($transactionId)
            ->setTransactionId($transactionId)
            ->setIsTransactionClosed(true)
            ->setShouldCloseParentTransaction(true);

        $payment->setParentTransactionId($payment->getTransactionId());
        $transaction = $payment->addTransaction(
            \Magento\Sales\Model\Order\Payment\Transaction::TYPE_ORDER,
            null,
            true,
            ""
        );
        $transaction->setIsClosed(true);
        $transaction->save();
        $order->save();
        $this->_orderRepository->save($order);
    }
}
