<?php

/**
 * Webkul Software
 *
 * @category  Webkul
 * @package   Webkul_BitcoinCryptoPayment
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\BitcoinCryptoPayment\ViewModel;

use Magento\Framework\Pricing\Helper\Data;
use Magento\Directory\Model\Currency as CurrencyModel;

class OrderInformation implements \Magento\Framework\View\Element\Block\ArgumentInterface
{
    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    protected $resourceconnection;

    /**
     * @var \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto
     */
    protected $configsection;

    /**
     * @var Data
     */
    protected $priceHelper;

    /**
     * @var \Magento\Sales\Api\Data\OrderInterface
     */
    protected $order;

    /**
     * @var CurrencyModel
     */
    protected $currencyModel;

    /**
     * Constructor
     *
     * @param \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory
     * @param \Magento\Framework\App\ResourceConnection $resourceconnection
     * @param \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto $configsection
     * @param \Magento\Sales\Api\Data\OrderInterface $order
     * @param Data $priceHelper
     * @param CurrencyModel $currencyModel
     */
    public function __construct(
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory,
        \Magento\Framework\App\ResourceConnection $resourceconnection,
        \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto $configsection,
        \Magento\Sales\Api\Data\OrderInterface $order,
        Data $priceHelper,
        CurrencyModel $currencyModel
    ) {
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->resourceconnection = $resourceconnection;
        $this->configsection = $configsection;
        $this->order = $order;
        $this->priceHelper = $priceHelper;
        $this->currencyModel = $currencyModel;
    }

    /**
     * Get Order Details For Crypto
     *
     * @param int $orderId
     * @return \Magento\Sales\Model\Order
     */
    public function getOrderDetailsForCryro($orderId)
    {
        $crypto_payments_transactions = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments_transactions');
        $crypto_payments = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments');
        $orders = $this->orderCollectionFactory->create();
        $orders->addAttributeToSelect('entity_id');
        $orders->addAttributeToSelect('grand_total');
        $orders->addAttributeToSelect('order_currency_code');
        $orders->addAttributeToSelect('crypto_payment_method_id');
        $orders->addAttributeToSelect('status');
        $orders->addFieldToFilter('entity_id', ['in' => $orderId]);
        $orders->getSelect()->join(
            ['crypto_transaction' => $crypto_payments_transactions],
            'main_table.entity_id = crypto_transaction.sales_order_id',
            ['order_id' => 'main_table.entity_id',
                'reminder_count' => 'crypto_transaction.reminder_count',
                'crypto_amount' => 'crypto_transaction.crypto_amount',
                'transaction_number' => 'crypto_transaction.transaction_number']
        );
        $orders->getSelect()->join(
            ['crypto_payment_methods' => $crypto_payments],
            'crypto_transaction.crypto_payment_method_id = crypto_payment_methods.entity_id',
            ['wallet_name' => 'crypto_payment_methods.name',
                'wallet_unit' => 'crypto_payment_methods.unit',
                'wallet_address' => 'crypto_payment_methods.address',
                'is_tag_required' => 'crypto_payment_methods.is_tag_required',
                'destination_tag_memo' => 'crypto_payment_methods.destination_tag_memo']
        );
        return $orders->getFirstItem();
    }

    /**
     * Cancel Days
     *
     * @return int
     */
    public function getCancelInDays()
    {
        $cancelledDays = (int) $this->configsection->cancelOrderTimePeriod();
        return $cancelledDays;
    }

    /**
     * Show QR
     *
     * @return int
     */
    public function getShowQrCode()
    {
        $show = (int) $this->configsection->isDisplayWalletAddressOrCode();
        return $show;
    }

    /**
     * Get CancelOrder
     *
     * @return int
     */
    public function getCancelOrderEnable()
    {
        $enable = (int) $this->configsection->cancelOrder();
        return $enable;
    }

    /**
     * Get Formatted Amount
     *
     * @param float $amount
     * @return string
     */
    public function getFormattedAmount($amount)
    {
        return $this->priceHelper->currency($amount, true, false);
    }

    /**
     * Add Currency Format
     *
     * @param float $amount
     * @param string $currencySymbol
     * @param int $precision
     * @return string
     */
    public function currencyFormat($amount, $currencySymbol, $precision = 2)
    {
        return $this->currencyModel->format($amount, ['symbol' => $currencySymbol, 'precision'=> $precision], false);
    }

    /**
     * Get OrderMainId
     *
     * @param int $orderId
     * @return int
     */
    public function getOrderMainId($orderId)
    {
        $orders = $this->orderCollectionFactory->create();
        $orders->addAttributeToSelect('entity_id');
        $orders->addFieldToFilter('increment_id', ['in' => $orderId]);
        return $orders->getFirstItem()->getId();
    }

    /**
     * Get Order Object
     *
     * @param int $orderId
     * @return \Magento\Sales\Model\Order
     */
    public function getOrderObject($orderId)
    {
        return $this->order->load($orderId);
    }
}
