<?php

/**
 * Webkul Software
 *
 * @category  Webkul
 * @package   Webkul_BitcoinCryptoPayment
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\BitcoinCryptoPayment\Model;

use Magento\Checkout\Model\ConfigProviderInterface;
use Magento\Framework\Pricing\Helper\Data as PriceHelper;
use Magento\Payment\Helper\Data as PaymentHelper;
use Webkul\BitcoinCryptoPayment\Helper\Data as CryptoHelper;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto;

class BitcoinCryptoPaymentConfigProvider implements ConfigProviderInterface
{
    /**
     * @var string
     */
    protected $methodCode = BitcoinCrypto::PAYMENT_METHOD_CRYPTOBITCOIN_CODE;

    /**
     * @var BitcoinCrypto
     */
    protected $method;

    /**
     * @var CryptoHelper
     */
    protected $cryptoHelper;

    /**
     * @var PriceHelper
     */
    protected $priceHelper;

    /**
     * @param PaymentHelper $paymentHelper
     * @param CryptoHelper $cryptoHelper
     * @param PriceHelper $priceHelper
     */
    public function __construct(
        PaymentHelper $paymentHelper,
        CryptoHelper $cryptoHelper,
        PriceHelper $priceHelper
    ) {
        $this->cryptoHelper = $cryptoHelper;
        $this->method = $paymentHelper->getMethodInstance($this->methodCode);
        $this->priceHelper = $priceHelper;
    }

    /**
     * @inheritdoc
     */
    public function getConfig()
    {
        return $this->method->isAvailable() ? [
            'payment' => [
                'bitcoincryptopayment' => [
                    'listType' => $this->method->getPaymentListingType(),
                    'paymentTemplate' => $this->getHtmlTemplateArrayToDisplay(),
                    'wkShowQR' => $this->method->isDisplayWalletAddressOrCode(),
                    'sortedMethods' => $this->getSortedPaymentMethods(),
                ],
            ],
        ] : [];
    }

    /**
     * Get available crypto payment methods
     *
     * @return array
     */
    public function getAvailableCryptoPaymentMethods()
    {
        $paymentDetails = $this->cryptoHelper->getCryptoPaymentMethodsForOrder();

        if ($paymentDetails) {
            $totalOrderAmt = (float) $this->cryptoHelper->getCurrentQuote()->getGrandTotal();
            $currencySign = $this->cryptoHelper->getQuoteCurrency();
            foreach ($paymentDetails as $k => &$method) {
                $exchangeRate = $this->cryptoHelper->getExchangeRate(
                    $totalOrderAmt,
                    $currencySign,
                    $method['unit'],
                    $method['decimal_places']
                );
                if ($exchangeRate) {
                    $method['exchaged_amount'] = $exchangeRate;
                } else {
                    unset($paymentDetails[$k]);
                }
            }

            $updatedArray = array_values($paymentDetails);
            return $updatedArray;
        }
        return [];
    }

    /**
     * GetTemplate Array
     *
     * @return array
     */
    public function getHtmlTemplateArrayToDisplay()
    {
        $htmlDataArray = [];
        $methodsArray = $this->getAvailableCryptoPaymentMethods();
        foreach ($methodsArray as $key => $method) {
            $totalOrderAmt = $this->cryptoHelper->getCurrentQuote()->getGrandTotal();
            $formattedAmount = $this->priceHelper->currency($totalOrderAmt, true, false);
            $htmlDataArray[] = [
                'entity_id' => $method['entity_id'],
                'exchaged_amount' => $method['exchaged_amount'],
                'unit' => $method['unit'],
                'isDisplayQrCode' => $this->method->isDisplayWalletAddressOrCode(),
                'address' => $method['address'],
                'formattedAmount' => $formattedAmount,
                'iscancelOrder' => $this->method->cancelOrder(),
                'cancelDaysNumber' => $this->method->cancelOrderTimePeriod()
            ];
        }
        return $htmlDataArray;
    }

    /**
     * Get Sorted Payment Methods
     *
     * @return array
     */
    public function getSortedPaymentMethods()
    {
        $methods = $this->getAvailableCryptoPaymentMethods();
        $sortedArray = [];
        foreach ($methods as $key => $code) {
            $sortedArray[$code['entity_id']] = $code;
        }

        return $sortedArray;
    }
}
