<?php

/**
 * Webkul Software
 *
 * @category  Webkul
 * @package   Webkul_BitcoinCryptoPayment
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\BitcoinCryptoPayment\Helper;

use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Sales\Model\Order\Email\Sender\OrderCommentSender;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCryptoPayment as CryptoCollection;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCryptoTransactions;

class Data extends AbstractHelper
{
    public const WK_EXCHANGE_RATE_URL = 'https://api.coinbase.com/v2/exchange-rates';

    /**
     * @var CryptoCollection
     */
    protected $cryptocollection;

    /**
     * @var \Magento\Customer\Model\SessionFactory
     */
    protected $_customerSession;

    /**
     * @var \Magento\Checkout\Model\SessionFactory
     */
    protected $checkoutSession;

    /**
     * @var \Magento\Sales\Model\ResourceModel\Order\CollectionFactory
     */
    protected $orderCollectionFactory;

    /**
     * @var \Magento\Framework\App\ResourceConnection
     */
    protected $resourceconnection;

    /**
     * @var \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto
     */
    protected $configsection;

    /**
     * @var BitcoinCryptoTransactions
     */
    protected $cryptoTranactions;

    /**
     * @var \Webkul\BitcoinCryptoPayment\Logger\Logger
     */
    protected $logger;

    /**
     * @var Email
     */
    protected $emailhelper;

    /**
     * @var OrderCommentSender
     */
    protected $orderCommentSender;

    /**
     * @var \Magento\Sales\Api\OrderManagementInterface
     */
    protected $orderManagement;

    /**
     * @var \Magento\Framework\HTTP\ClientInterface
     */
    protected $client;

    /**
     * @var \Magento\Framework\Json\Helper\Data
     */
    protected $jsonHelper;

    /**
     * @var \Magento\Framework\App\Config\ScopeConfigInterface
     */
    protected $scopeConfig;

    /**
     * @var \Magento\Framework\Encryption\EncryptorInterface
     */
    protected $_encryptor;

    /**
     * Initialise helper class
     *
     * @param Context $context
     * @param CryptoCollection $cryptocollection
     * @param \Magento\Customer\Model\SessionFactory $customerSession
     * @param \Magento\Checkout\Model\SessionFactory $checkoutSession
     * @param \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory
     * @param \Magento\Framework\App\ResourceConnection $resourceconnection
     * @param \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto $configsection
     * @param BitcoinCryptoTransactions $cryptoTranactions
     * @param \Webkul\BitcoinCryptoPayment\Logger\Logger $logger
     * @param Email $emailhelper
     * @param OrderCommentSender $orderCommentSender
     * @param \Magento\Sales\Api\OrderManagementInterface $orderManagement
     * @param \Magento\Framework\HTTP\ClientInterface $client
     * @param \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
     * @param \Magento\Framework\Json\Helper\Data $jsonHelper
     * @param \Magento\Framework\Encryption\EncryptorInterface $encryptor
     */
    public function __construct(
        Context $context,
        CryptoCollection $cryptocollection,
        \Magento\Customer\Model\SessionFactory $customerSession,
        \Magento\Checkout\Model\SessionFactory $checkoutSession,
        \Magento\Sales\Model\ResourceModel\Order\CollectionFactory $orderCollectionFactory,
        \Magento\Framework\App\ResourceConnection $resourceconnection,
        \Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto $configsection,
        BitcoinCryptoTransactions $cryptoTranactions,
        \Webkul\BitcoinCryptoPayment\Logger\Logger $logger,
        Email $emailhelper,
        OrderCommentSender $orderCommentSender,
        \Magento\Sales\Api\OrderManagementInterface $orderManagement,
        \Magento\Framework\HTTP\ClientInterface $client,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig,
        \Magento\Framework\Json\Helper\Data $jsonHelper,
        \Magento\Framework\Encryption\EncryptorInterface $encryptor
    ) {
        $this->cryptocollection = $cryptocollection;
        $this->_customerSession = $customerSession;
        $this->checkoutSession = $checkoutSession;
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->resourceconnection = $resourceconnection;
        $this->configsection = $configsection;
        $this->cryptoTranactions = $cryptoTranactions;
        $this->logger = $logger;
        $this->emailhelper = $emailhelper;
        $this->orderCommentSender = $orderCommentSender;
        $this->orderManagement = $orderManagement;
        $this->client = $client;
        $this->scopeConfig = $scopeConfig;
        $this->jsonHelper = $jsonHelper;
        $this->_encryptor = $encryptor;
        parent::__construct($context);
    }

    /**
     * Get Payment Collection
     *
     * @return CryptoCollection
     */
    public function getCryptoPaymentCollection()
    {
        return $this->cryptocollection->getCollection()->setOrder('sort_order', 'ASC');
    }

    /**
     * Get Customer Group Id
     *
     * @return int
     */
    public function getCustomerGroupId()
    {
        $customerGroup = 0;
        if ($this->_customerSession->create()->isLoggedIn()) {
            $customerGroup = $this->_customerSession->create()->getCustomer()->getGroupId();
        }

        return $customerGroup;
    }

    /**
     * Get QuoteCurrency
     *
     * @return string
     */
    public function getQuoteCurrency()
    {
        $quote = $this->checkoutSession->create()->getQuote();
        return $quote->getCurrency()->getQuoteCurrencyCode() ?? 'USD';
    }

    /**
     * Get Quote Carrier
     *
     * @return string
     */
    public function getQuoteCarrier()
    {
        $quote = $this->checkoutSession->create()->getQuote();
        $shippingMethod = $quote->getShippingAddress()->getShippingMethod();
        return $shippingMethod;
    }

    /**
     * Get Current Quote
     *
     * @return \Magento\Quote\Model\Quote
     */
    public function getCurrentQuote()
    {
        $quote = $this->checkoutSession->create()->getQuote();
        return $quote;
    }

    /**
     * Get currency exchange rate
     *
     * @param float $amount
     * @param string $from
     * @param string $to
     * @param int $decimalPlaces
     *
     * @return boolean|float
     */
    public function getExchangeRate($amount = 1, $from = 'USD', $to = 'BTC', $decimalPlaces = 7)
    {
        try {
            $requestUrl = self::WK_EXCHANGE_RATE_URL . '?currency=' . strtoupper($from);
            $this->client->get($requestUrl);
            $responseJson = $this->client->getBody();
            $response = $this->jsonHelper->jsonDecode($responseJson);
            if ($response && isset($response['data']['rates'])) {
                return round($amount * $response['data']['rates'][$to], $decimalPlaces);
            }
        } catch (\Exception $e) {
            $this->logger->info('Get Exchange Rate Error : ' . $e->getMessage());
            return false;
        }
        return false;
    }

    /**
     * Send Reminder Email
     *
     * @return void
     */
    public function sendReminderEmail()
    {
        $reminderTime = (int) $this->configsection->reminderInterval();
        $totalTime = $reminderTime;
        $awaitingOrders = $this->getAllAwaitingOrder($totalTime, true);
        if ($awaitingOrders) {
            foreach ($awaitingOrders as $order) {
                $orderTime = ((int) $order->getReminderCount() + 1) * $reminderTime;
                $currentDate = date('Y-m-d H:i:s');
                $currentTimeStamp = strtotime($currentDate);
                $endTimeStamp = strtotime($order->getCreatedAt());
                $endTimeStamp += 24 * 3600 * $orderTime;
                if ($currentTimeStamp >= $endTimeStamp) {
                    $trans = $this->cryptoTranactions->load($order->getId(), 'sales_order_id');
                    $trans->setReminderCount((int) $trans->getReminderCount() + 1);
                    $trans->save();
                    $this->sendReminderMailToCustomer($order);
                    $txt = 'Reminder Send successfuly for order id-' . $order->getId();
                    $this->logger->info($txt);
                }
            }
        }
    }

    /**
     * Get All Awaiting Order
     *
     * @param int $days
     * @param boolean $reminder
     *
     * @return \Magento\Sales\Model\ResourceModel\Order\Collection
     */
    public function getAllAwaitingOrder($days, $reminder = false)
    {
        $currentTimeStamp = date('Y-m-d H:i:s', strtotime('- ' . $days . ' day'));
        $crypto_payments_transactions = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments_transactions');
        $orders = $this->orderCollectionFactory->create();
        $orders->getSelect()->join(
            ['crypto_transaction' => $crypto_payments_transactions],
            'main_table.entity_id = crypto_transaction.sales_order_id',
            ['created_at_order' => 'main_table.created_at',
                'order_id' => 'main_table.entity_id',
                'reminder_count' => 'crypto_transaction.reminder_count']
        );
        $orders->getSelect()->where('main_table.created_at < "' . $currentTimeStamp . '"');
        $orders->getSelect()->where('main_table.status = "awaiting_for_crypto_payment"');
        if ($reminder) {
            $orders->getSelect()->where('crypto_transaction.reminder_count < "' .
                (int) $this->configsection->numberOfReminder() . '"');
        }

        return $orders;
    }

    /**
     * Send Reminder
     *
     * @param \Magento\Sales\Model\Order $order
     *
     * @return void
     */
    public function sendReminderMailToCustomer($order)
    {
        $crypto_payments_transactions = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments_transactions');
        $crypto_payments = $this->resourceconnection
            ->getTableName('webkul_bitcoin_crypto_payments');
        $payment = $order->getPayment();
        $method = $payment->getMethodInstance();
        $paymentTitle = $method->getTitle();
        $orders = $this->orderCollectionFactory->create();
        $orders->addFieldToFilter('entity_id', ['in' => $order->getId()]);
        $orders->getSelect()->joinLeft(
            ['crypto_transaction' => $crypto_payments_transactions],
            'main_table.entity_id = crypto_transaction.sales_order_id',
            ['order_id' => 'main_table.entity_id',
                'reminder_count' => 'crypto_transaction.reminder_count',
                'crypto_amount' => 'crypto_transaction.crypto_amount',
                'transaction_number' => 'crypto_transaction.transaction_number']
        );

        $orders->getSelect()->joinLeft(
            ['crypto_payment' => $crypto_payments],
            'crypto_payment.entity_id = crypto_transaction.crypto_payment_method_id',
            ['wallet_address' => 'crypto_payment.address',
                'wallet_name' => 'crypto_payment.name',
                'wallet_unit' => 'crypto_payment.unit',
                'is_tag_required' => 'crypto_payment.is_tag_required',
                'destination_tag_memo' => 'crypto_payment.destination_tag_memo']
        );
        $orderDetails = $orders->getFirstItem();

        // Crypto payment confirmed by customer
        $emailTemplateVariables = [
            'customer_name' => $order->getCustomerName() ?? 'Guest',
            'order_incrementid' => $order->getIncrementId(),
            'wallet_name' => $paymentTitle . '(' . $orderDetails->getWalletName() . ')',
            'wallet_address' => $orderDetails->getWalletAddress(),
            'crypto_amount' => $orderDetails->getCryptoAmount() . ' ' . $orderDetails->getWalletUnit(),
            'order_amount' => $this->emailhelper->currencyFormat(
                $order->getGrandTotal(),
                $order->getOrderCurrencyCode()
            ),
            'is_tag_required' => $orderDetails->getIsTagRequired() ?? "",
            'destination_tag_memo' => $orderDetails->getDestinationTagMemo() ?? "NA",
        ];

        $receiverInfo = [$order->getCustomerEmail()];
        $template = 'webkul/custom/email_template_reminder';
        $this->emailhelper->sendEmail($emailTemplateVariables, $receiverInfo, $template);
    }

    /**
     * Automatic Cancel Order
     *
     * @return void
     */
    public function automaticCancelOrder()
    {
        $cancelledDays = (int) $this->configsection->cancelOrderTimePeriod();
        if ($cancelledDays) {
            $cancelledOrders = $this->getOrderListForCancel($cancelledDays);
            if ($cancelledOrders) {
                foreach ($cancelledOrders as $order) {
                    try {
                        $this->orderManagement->cancel($order->getId());
                        if ($order->getState() == 'canceled') {
                            $this->orderCommentSender->send($order, true);
                        }
                        $this->logger->info('Automatic order cancel start');

                        $this->logger->info($order->getIncrementId());
                        $this->logger->info('Automatic order cancel end');
                    } catch (\Exception $e) {
                        $this->logger->info('Automatic order cancel error');

                        $this->logger->info($order->getIncrementId());

                        $this->logger->info($e->getMessage());
                        $this->logger->info('Automatic order cancel error');
                    }
                }
            }
        }
    }

    /**
     * Get Order list
     *
     * @param int $days
     * @return \Magento\Sales\Model\ResourceModel\Order\Collection
     */
    public function getOrderListForCancel($days)
    {
        $currentTimeStamp = date('Y-m-d H:i:s', strtotime('- ' . $days . ' day'));
        $orders = $this->orderCollectionFactory->create();
        $orders->addFieldToFilter('created_at', ['gteq' => $currentTimeStamp]);
        $orders->addFieldToFilter('status', ['in' => "awaiting_for_crypto_payment"]);
        return $orders;
    }

    /**
     * Get crypto payment methods for order
     *
     * @return array
     */
    public function getCryptoPaymentMethodsForOrder()
    {
        $paymentDetails = $this->getCryptoPaymentCollection()->getData();
        foreach ($paymentDetails as $key => $paymentMethod) {

            if ($paymentMethod['status'] == 0) {
                unset($paymentDetails[$key]);
                continue;
            }
            if ($paymentMethod['customer_groups'] != '' &&
                $accessGroups = explode(',', $paymentMethod['customer_groups'] ?? '')) {
                $customerGroups = $this->getCustomerGroupId();

                if (!in_array($customerGroups, $accessGroups)) {
                    unset($paymentDetails[$key]);
                    continue;
                }
            }

            if ($paymentMethod['currency'] != '' &&
                $accessCurrencies = explode(',', $paymentMethod['currency'] ?? '')) {
                if (!in_array($this->getQuoteCurrency(), $accessCurrencies)) {
                    unset($paymentDetails[$key]);
                    continue;
                }
            }

            if (!$this->getCurrentQuote()->isVirtual()) {
                if ($paymentMethod['carrier'] != ''
                    && $accessCarriers = explode(',', $paymentMethod['carrier'] ?? '')) {
                    if (!in_array($this->getQuoteCarrier(), $accessCarriers)) {
                        unset($paymentDetails[$key]);
                        continue;
                    }
                }
            }
        }

        return $paymentDetails;
    }
}
