<?php

/**
 * Webkul Software
 *
 * @category  Webkul
 * @package   Webkul_BitcoinCryptoPayment
 * @author    Webkul Software Private Limited
 * @copyright Webkul Software Private Limited (https://webkul.com)
 * @license   https://store.webkul.com/license.html
 */

namespace Webkul\BitcoinCryptoPayment\Controller\Crypto;

use Magento\Framework\Controller\ResultFactory;
use Magento\Directory\Model\Currency as CurrencyModel;
use Webkul\BitcoinCryptoPayment\Helper\Data as CryptoHelper;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCrypto as PaymentModel;
use Webkul\BitcoinCryptoPayment\Model\BitcoinCryptoPayment as CryptoCollection;

/**
 * Class Request
 * Generate request data to create order and proceed payment
 */
class Request extends \Magento\Framework\App\Action\Action
{
    /**
     * @var CryptoHelper
     */
    protected $cryptoHelper;

    /**
     * @var CurrencyModel
     */
    protected $currencyModel;

    /**
     * @var PaymentModel
     */
    protected $paymentmodel;

    /**
     * @var CryptoCollection
     */
    protected $cryptocollection;

    /**
     * @var \Magento\Quote\Model\QuoteRepository
     */
    protected $quoteRepository;

    /**
     * @var \Magento\Quote\Model\QuoteIdMaskFactory
     */
    protected $quoteIdMaskFactory;

    /**
     * Constructor
     *
     * @param \Magento\Framework\App\Action\Context $context
     * @param CryptoHelper $cryptoHelper
     * @param CurrencyModel $currencyModel
     * @param PaymentModel $paymentmodel
     * @param CryptoCollection $cryptocollection
     * @param \Magento\Quote\Model\QuoteRepository $quoteRepository
     * @param \Magento\Quote\Model\QuoteIdMaskFactory $quoteIdMaskFactory
     */
    public function __construct(
        \Magento\Framework\App\Action\Context $context,
        CryptoHelper $cryptoHelper,
        CurrencyModel $currencyModel,
        PaymentModel $paymentmodel,
        CryptoCollection $cryptocollection,
        \Magento\Quote\Model\QuoteRepository $quoteRepository,
        \Magento\Quote\Model\QuoteIdMaskFactory $quoteIdMaskFactory
    ) {
        $this->cryptoHelper = $cryptoHelper;
        $this->currencyModel = $currencyModel;
        $this->paymentmodel = $paymentmodel;
        $this->cryptocollection = $cryptocollection;
        $this->quoteRepository = $quoteRepository;
        $this->quoteIdMaskFactory = $quoteIdMaskFactory;
        parent::__construct($context);
    }
    /**
     * Execute function
     *
     * @return \Magento\Framework\Controller\ResultInterface
     */
    public function execute()
    {
        $crypto_id = $this->getRequest()->getParam('crypto_id');
        $quoteid = $this->getRequest()->getParam('quoteid');
        $customerIsLogin = $this->getRequest()->getParam('login');
        $crypto_currency = $this->getRequest()->getParam('crypto_currency');
        $conversion_rate = $this->getRequest()->getParam('conversion_rate');
        $crypto_amount = $this->getRequest()->getParam('crypto_amount');

        $paymentDetails = $this->cryptocollection->load($crypto_id);
        $responseContent = [
            'success' => false,
        ];

        if ($paymentDetails) {

            $currentQuote = $this->cryptoHelper->getCurrentQuote();

            $totalOrderAmt = (float) $currentQuote->getGrandTotal();
            $currencySign = $this->cryptoHelper->getQuoteCurrency();
            $exchangeRate = $this->cryptoHelper->getExchangeRate(
                $totalOrderAmt,
                $currencySign,
                $paymentDetails->getUnit(),
                $paymentDetails->getDecimalPlaces()
            );

            $cartId = $quoteid ?? 0;
            $isLogin = $customerIsLogin == 0 ? false : true;

            if (!$isLogin) {
                $quoteIdMask = $this->quoteIdMaskFactory->create()->load($cartId, 'masked_id');
                if ($quoteIdMask->getId()) {
                    $cartId = $quoteIdMask->getQuoteId();
                }
            }

            $quote = $this->quoteRepository->get($cartId);
            $quote->setData('crypto_currency', $crypto_currency);
            $quote->setData('conversion_rate', $conversion_rate);
            $quote->setData('crypto_payment_method_id', $crypto_id);
            $quote->setData('crypto_amount', $crypto_amount);
            $this->quoteRepository->save($quote);

            $method = $paymentDetails->getData();
            $totalOrderAmt = (float) $currentQuote->getGrandTotal();
            $currencySign = $this->cryptoHelper->getQuoteCurrency();
            $exchangeRate = $this->cryptoHelper->getExchangeRate(
                $totalOrderAmt,
                $currencySign,
                $method['unit'],
                $method['decimal_places']
            );

            $currencyCode = $currentQuote->getQuoteCurrencyCode();
            $currencySymbol = $this->currencyModel->load($currencyCode)->getCurrencySymbol();
            $precision = 2;

            $formattedAmount = $this->currencyModel->format(
                $totalOrderAmt,
                ['symbol' => $currencySymbol, 'precision'=> $precision],
                false,
                false
            );
            $htmlDataArray = ['entity_id' => $method['entity_id'],
                'exchaged_amount' => $exchangeRate,
                'unit' => $method['unit'],
                'isDisplayQrCode' => $this->paymentmodel->isDisplayWalletAddressOrCode(),
                'address' => $method['address'],
                'formattedAmount' => $formattedAmount,
                'iscancelOrder' => $this->paymentmodel->cancelOrder(),
                'cancelDaysNumber' => $this->paymentmodel->cancelOrderTimePeriod(),
                'isShowTag' => $method['is_tag_required'],
                'dTag' => $method['destination_tag_memo'],
            ];

            $responseContent = [
                'success' => true,
                'walletAddress' => $paymentDetails->getAddress(),
                'paymentDetails' => $paymentDetails->getData(),
                'exchangeRate' => $exchangeRate,
                'psOrderTotal' => $formattedAmount,
                'showQR' => (bool) $this->paymentmodel->isDisplayWalletAddressOrCode(),
                'cancel_order_enabled' => (bool) $this->paymentmodel->cancelOrder(),
                'cancel_in_days' => $this->paymentmodel->cancelOrderTimePeriod(),
                'htmlContentArray' => $htmlDataArray,
            ];
        }

        $response = $this->resultFactory->create(ResultFactory::TYPE_JSON);
        $response->setData($responseContent);
        return $response;
    }
}
