<?php

namespace Capdeal\Ads\Controller\Post;

use Magento\Framework\App\Action\Action;
use Magento\Framework\App\Action\Context;
use Magento\Catalog\Model\ProductFactory;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Catalog\Model\CategoryRepository;
use Magento\Customer\Model\Session;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Filesystem;
use Magento\MediaStorage\Model\File\UploaderFactory;
use Magento\Indexer\Model\Processor;

class Save extends Action
{
    protected $productFactory;
    protected $storeManager;
    protected $categoryRepository;
    protected $customerSession;
    protected $filesystem;
    protected $uploaderFactory;
    protected $indexerProcessor;

    public function __construct(
        Context $context,
        ProductFactory $productFactory,
        StoreManagerInterface $storeManager,
        CategoryRepository $categoryRepository,
        Session $customerSession,
        Filesystem $filesystem,
        UploaderFactory $uploaderFactory,
        Processor $indexerProcessor
    ) {
        parent::__construct($context);
        $this->productFactory = $productFactory;
        $this->storeManager = $storeManager;
        $this->categoryRepository = $categoryRepository;
        $this->customerSession = $customerSession;
        $this->filesystem = $filesystem;
        $this->uploaderFactory = $uploaderFactory;
        $this->indexerProcessor = $indexerProcessor;
    }

    public function execute()
    {
        if (!$this->customerSession->isLoggedIn()) {
            $this->messageManager->addErrorMessage(__('You must be logged in to create an ad.'));
            return $this->_redirect('customer/account/login');
        }

        $customerId = $this->customerSession->getCustomerId();
        $data = $this->getRequest()->getPostValue();

        if ($data) {
            try {
                $product = $this->productFactory->create();
                $product->setSku('ad_' . time());
                $product->setName($data['title']);
                $product->setTypeId(\Magento\Catalog\Model\Product\Type::TYPE_SIMPLE);
                $product->setPrice($data['price']);
                $product->setVisibility(\Magento\Catalog\Model\Product\Visibility::VISIBILITY_BOTH);
                $product->setStatus(\Magento\Catalog\Model\Product\Attribute\Source\Status::STATUS_ENABLED);
                $product->setAttributeSetId(4);
                $product->setStockData(['use_config_manage_stock' => 1, 'qty' => 1, 'is_in_stock' => 1]);
                $product->setDescription($data['description']);
                $product->setWebsiteIds([1]); 
             
                

                $product->setData('customer_whatsapp', $data['customer_whatsapp']);
                $product->setData('customer_call', $data['customer_call']);
                $product->setData('product_customer', $customerId);
                $urlKey = strtolower(str_replace(' ', '-', $data['title'])) . '-' . time();
                $product->setUrlKey($urlKey);
                $product->save();

                // Upload Main Image
                if (isset($_FILES['main_image']['name']) && $_FILES['main_image']['name'] != '') {
                    $mainImagePath = $this->uploadImage('main_image');
                    if ($mainImagePath) {
                        $product->addImageToMediaGallery($mainImagePath, ['image', 'small_image', 'thumbnail'], false, false);
                    }
                }

                // Upload Additional Images
                if (isset($_FILES['additional_images']['name']) && is_array($_FILES['additional_images']['name'])) {
                    foreach ($_FILES['additional_images']['name'] as $key => $value) {
                        if ($_FILES['additional_images']['name'][$key] != '') {
                            $additionalImagePath = $this->uploadAdditionalImage('additional_images', $key);
                            if ($additionalImagePath) {
                                $product->addImageToMediaGallery($additionalImagePath, null, false, false);
                            }
                        }
                    }
                }

                $product->save();

                if (!empty($data['category_id'])) {
                    try {
                        $category = $this->categoryRepository->get($data['category_id']); // Validate category
                        $product->setCategoryIds([$data['category_id']]);
                        $product->save(); // Save again after setting category
                    } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                        $this->messageManager->addErrorMessage(__('Invalid Category Selected.'));
                    }
                }
                

                // **🔹 Run Reindex After Saving Product**
                $this->indexerProcessor->reindexAll();

                $this->messageManager->addSuccessMessage(__('Your ad has been successfully created and indexed.'));
            } catch (\Exception $e) {
                $this->messageManager->addErrorMessage(__('Error: ' . $e->getMessage()));
            }
        }

        return $this->_redirect('ads/index/index');
    }

    private function uploadImage($imageField)
    {
        try {
            $mediaDirectory = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath('catalog/product/');
            $uploader = $this->uploaderFactory->create(['fileId' => $imageField]);
            $uploader->setAllowedExtensions(['jpg', 'jpeg', 'png', 'gif']);
            $uploader->setAllowRenameFiles(true);
            $uploader->setFilesDispersion(true);
            $result = $uploader->save($mediaDirectory);
            return 'catalog/product' . $result['file'];
        } catch (\Exception $e) {
            return null;
        }
    }

    private function uploadAdditionalImage($imageField, $key)
    {
        try {
            $mediaDirectory = $this->filesystem->getDirectoryRead(DirectoryList::MEDIA)->getAbsolutePath('catalog/product/');
            $file = [
                'name' => $_FILES[$imageField]['name'][$key],
                'type' => $_FILES[$imageField]['type'][$key],
                'tmp_name' => $_FILES[$imageField]['tmp_name'][$key],
                'error' => $_FILES[$imageField]['error'][$key],
                'size' => $_FILES[$imageField]['size'][$key]
            ];
            $uploader = $this->uploaderFactory->create(['fileId' => $file]);
            $uploader->setAllowedExtensions(['jpg', 'jpeg', 'png', 'gif']);
            $uploader->setAllowRenameFiles(true);
            $uploader->setFilesDispersion(true);
            $result = $uploader->save($mediaDirectory);
            return 'catalog/product' . $result['file'];
        } catch (\Exception $e) {
            return null;
        }
    }
}
